﻿/*	VERSION:	1.5
1.5		make init() a pure function,  which reveals exactly what data it depends on,  and what data it is creating.
			make_loopingImage() returns data including "wrapCoords"
1.4		non-wrapping bottom edge-limit is based upon rotation-point.  (downward scrolling stops when the rotation point hits the bottom of the image)
1.3		Made image-wrapping optional.   When wrapping is turned off,  then the image stops scrolling when edges hit the center,  and then slides the center along the edge.
1.2		added rot_speed parameter to make_loopingImage()   and   to the output object as a writable property
1.1		renamed init function to make_loopingImage()  (used to be make_imageLoop()

INPUT:		worldMap_bitmap			Picture to use for this map
					cropYScale						Specify how to crop the resulting image  (1.0 = display full safe-height image, rotates around center)  (0.5 = display the image at 1/2 height, rotates around the bottom)  (0.6 = display a little over half the height of the image, rotates around a point slightly above the bottom)

OUTPUT:		loop()							Used to update the moving map image / pause it (by not calling loop()
					vel {}							Used to modify the position
						x
						y
						radian
					pos {}							Used to read / set the current position & angle on the map
						x
						y
						radian
					output_pic					Used to display the current area of the world map  (updated by loop()

																																																	
USAGE: 
	#include "functions/loopingImage.as"
	var rot_speed = Math.PI/2/2 /8;	// radians to rotate per 1 px		~0.1 radians per 1px
	var pic_interface = make_loopingImage( src_pic, 1.0, rot_speed );
	
	function onEnterFrame(){
		// pic_interface.vel.x = vel.x;										// pan
		pic_interface.vel.radian = vel.x;								// rotate
		pic_interface.vel.y = vel.y;
		pic_interface.loop();
	}// loop()
*/
function make_loopingImage( worldMap_bitmap, cropYScale, rot_speed, wrapImage, edgeOpaque, edgeColor ){
	// trace("version 1.5");
	
	// class shortcuts
	var BitmapData = flash.display.BitmapData;
	var Matrix = flash.geom.Matrix;
	var Rectangle = flash.geom.Rectangle;
	
	// resolve parameters
	var cropYScale = cropYScale || 1;
	if( wrapImage === undefined )		var wrapImage = true;
	if( isNaN(edgeColor) )					var edgeColor = 0;
	if( edgeOpaque === undefined )	var edgeOpaque = true;
	
	// ABORT if:  Not enough parameters to construct an imageLoop
	var canMake = (worldMap_bitmap instanceof BitmapData);
	if( !canMake ){
		throw new Error("Cannot make imageLoop, because it was not given a bitmap to use.");
		return;
	}
	
	
	// shared variables
	var vel,
			pos,
			crop_pic,
			clipRect, 
			output;
	var rot_speed;
	var angle_mat,
			angle_neg_mat,
			center_mat,
			center_neg_mat,
			crop_mat,
			display_mat,
			moveVec_mat,
			pattern_pic,
			position_neg_mat,
			wrapCoords;
	var pos_radian;
	var full_rect;
	
	// initialize vars
	if( isNaN(rot_speed)  )		rot_speed = .003;
	vel = {
		x:		0,
		y:		0,
		radian:	0
	}// vel {}
	pos = {
		// x			(property)
		// y			(property)
		// radian	(property)
	}// pos {}
	pos_radian = 0;		// track the current display-angle of the map, as radians.  (for getting & setting)
	
	function get_rot_speed(){
		return rot_speed;
	}// get_rot_speed()
	function set_rot_speed( newValue ){
		rot_speed = newValue;
	}// set_rot_speed()
	
	// initialize all internal working variables, create images
	var initData = init( worldMap_bitmap, cropYScale, edgeOpaque, wrapImage );
	pattern_pic = initData.pattern_pic;
	wrapCoords = initData.wrapBounds;
	full_rect = initData.full_rect;
	crop_mat = initData.crop_mat;
	crop_pic = initData.crop_pic;
	clipRect = initData.clipRect;
	angle_mat = initData.angle_mat;
	angle_neg_mat = initData.angle_neg_mat;
	position_neg_mat = initData.position_neg_mat;
	center_mat = initData.center_mat;
	center_neg_mat = initData.center_neg_mat;
	display_mat = initData.display_mat;
	moveVec_mat = initData.moveVec_mat;
	
	
	
	function applyVelocity(){
		if(vel.x){
			moveHorz( vel.x );
			wrapAround();
		}
		if(vel.y){
			moveVert( vel.y );
			wrapAround();
		}
		if(vel.radian){
			moveRot( vel.radian );
		}
	}// applyVelocity()
	
	
	
	// update loop
	function loop(){
		applyVelocity();
		updateImage();
	}// loop()
	
	
	// pos{}  (define properties)
	function pos_get_x(){
		var xx = -position_neg_mat.tx;
		return xx;
	}// get()
	function pos_set_x( newValue ){
		position_neg_mat.tx = -newValue;
	}// set()
	function pos_get_y(){
		var yy = -position_neg_mat.ty;
		return yy;
	}// get()
	function pos_set_y( newValue ){
		position_neg_mat.ty = -newValue;
	}// set()
	function pos_get_radian(){
		return pos_radian;
	}// get()
	function pos_set_radian( newValue ){
		pos_radian = newValue;
		limitPosRadian();
		angle_mat.identity();
		angle_neg_mat.identity();
		angle_mat.rotate( newValue );
		angle_neg_mat.rotate( -newValue );
	}// set()
	pos.addProperty("x", pos_get_x, pos_set_x);
	pos.addProperty("y", pos_get_y, pos_set_y);
	pos.addProperty("radian", pos_get_radian, pos_set_radian);		// uses radians
	
	
	//view_mc = this.createEmptyMovieClip("view_mc", 0);
	//view_mc.attachBitmap( crop_pic, 0, "never", false );
	
	//test_mc = this.createEmptyMovieClip("test_mc", 99);
	//test_mc.attachBitmap( pattern_pic, 0 );
	
	updateImage();
	output = {
		loop: loop,
		vel: vel,
		pos: pos,
		output_pic: crop_pic, 
		rot_speed: rot_speed, 
		wrapCoords: wrapCoords, 
		update: updateImage
	};// output {}
	
	output.addProperty("rot_speed", get_rot_speed, set_rot_speed);
	
	return output;
	
	
	
	
	/////////////////////////////////////////////////////////////////////////////////////
	/********************	init()
	Do the initial one-time setup.
	
	// input
	---
	
	// initalizes
	angle_mat
	angle_neg_mat
	center_mat
	center_neg_mat
	crop_mat
	crop_pic
	display_mat
	moveVec_mat
	pattern_pic
	position_neg_mat
	wrapCoords
	*******************/
	function init( texture_pic, cropYScale, edgeOpaque, wrapImage )
	{
		// Load the bitmap pattern
		//texture_pic = BitmapData.loadBitmap( "evermore.png" );
		
		// Create the display image.
		// define a safe crop area,  so no out-of-image areas will ever be seen
		var aspectRatio = texture_pic.width / texture_pic.height;
		//var rad_45 = 0.78539816339744830961566084581988;		// 45 degrees, in radians  (where the rectangle touches the safe-circle of a square image)
		//var x_scalar = Math.cos(rad_45);
		//var y_scalar = Math.sin(rad_45);	// omitted because, at 45 degrees, sin() and cos() have the same value
		var x_scalar = 0.707106781186548;		// hard-code the resulting value because it will never change
		var y_scalar = x_scalar * aspectRatio;		// x_scalar is used because, at 45 degrees, sin() and cos() have the same value
		var crop_x = Math.round((texture_pic.width/2) *(1-x_scalar));		// left offset
		var crop_y = Math.round((texture_pic.height/2) *(1-y_scalar));		// top offset
		var crop_width = Math.round((texture_pic.width/2 *x_scalar) *2);
		var crop_height = Math.round((texture_pic.height/2 *y_scalar) *2);
		var crop_mat = new Matrix();
		crop_mat.translate( -crop_x, -crop_y );
		var crop_pic = new BitmapData(crop_width, crop_height *cropYScale, !edgeOpaque, 0);
		// crop_pic = new BitmapData( texture_pic.width, texture_pic.height, !edgeOpaque, 0);		// used to test clipRect  (without clipRect, a very large image is displayed)
		var clipRect = new Rectangle( 0,0, crop_width,  (crop_height *cropYScale)  );
		
		// Create the main variables.
		var angle_mat = new Matrix();			var angle_neg_mat = new Matrix();
		var position_neg_mat = new Matrix();
			position_neg_mat.translate(-1,-1);
		var center_mat = new Matrix();		var center_neg_mat = new Matrix();
			center_mat.translate(-(texture_pic.width/2),-(texture_pic.height/2));	center_neg_mat.translate((texture_pic.width/2),(texture_pic.height/2));
		var display_mat = new Matrix();
		var moveVec_mat = new Matrix();
		
		// Prepare the bitmap pattern for looping
		var prepData;
		var texture_rect = new Rectangle( 0,0, texture_pic.width,texture_pic.height );
		var buffer = 0;
		if( wrapImage )			prepData = prepLoopingPattern( texture_pic, texture_rect, buffer );
		if( !wrapImage )		prepData = prepSinglePattern(  texture_pic, texture_rect, buffer, center_mat );
		var output = {
			pattern_pic: prepData.pattern_pic, 
			wrapBounds: prepData.wrapBounds, 
			full_rect: texture_rect.clone(), 
			crop_mat: crop_mat, 
			crop_pic: crop_pic, 
			clipRect: clipRect, 
			angle_mat: angle_mat, 
			angle_neg_mat: angle_neg_mat, 
			position_neg_mat: position_neg_mat, 
			center_mat: center_mat, 
			center_neg_mat: center_neg_mat, 
			display_mat: display_mat, 
			moveVec_mat: moveVec_mat
		}
		
		return output;
	}// init()
	
	
	
	/********************	prepLoopingPattern()
	Creates a loop-able image
	
	// input
	original_pic
	view_rect
	buffer
	
	// output
	pattern_pic
	wrapBounds
	*******************/
	function prepLoopingPattern( original_pic, view_rect, buffer )
	{
		//import flash.display.BitmapData;
		//import flash.geom.Rectangle;
		//import flash.geom.Point;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		var Point = flash.geom.Point;
		var xx,yy,ww,hh;
		var copy_rect, paste_p;
		var view_pic;
		var output, pattern_pic, wrapBounds;
		view_pic = new BitmapData( view_rect.width,view_rect.height, !edgeOpaque,0 );
		//buffer = 2;		// 2 pixel out from each edge
		
		
		
		// prepare the pattern for wrapping
		ww = original_pic.width +(buffer*2) +view_pic.width;
		hh = original_pic.height +(buffer*2) +view_pic.height;
		pattern_pic = new BitmapData( ww,hh, !edgeOpaque,0 );
		
		// copy the original into the prepped image
		copy_rect = new Rectangle( 0,0, original_pic.width, original_pic.height );
		paste_p = new Point( buffer, buffer );
		pattern_pic.copyPixels( original_pic, copy_rect, paste_p );
		
		
		
		// add buffer to the prepped image
		// // create right buffer
		// // // copy left part
		xx = buffer;
		yy = 0;
		ww = view_pic.width +buffer;
		hh = pattern_pic.height;
		copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the right
		xx = buffer +original_pic.width;
		yy = 0;
		paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		// // create bottom buffer
		// // // copy top part
		xx = 0;
		yy = buffer;
		ww = pattern_pic.width;
		hh = view_pic.height +buffer;
		copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the bottom
		xx = 0;
		yy = buffer +original_pic.height;
		paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		
		/*	
		// This part is probably only needed if the buffer is not Zero
		// // create left buffer
		// // // copy right part
		xx = original_pic.width;
		yy = 0;
		ww = buffer;
		hh = pattern_pic.height;
		copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the left
		xx = 0;
		yy = 0;
		paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		// // create top buffer
		// // // copy bottom part
		xx = 0;
		yy = original_pic.height;
		ww = pattern_pic.width;
		hh = buffer;
		copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the top
		xx = 0;
		yy = 0;
		paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		*/
		
		
		
		
		wrapBounds = {
			xMin: buffer,
			yMin: buffer,
			xMax: buffer +original_pic.width,
			yMax: buffer +original_pic.height
		};
		
		output = {
			pattern_pic: pattern_pic,
			wrapBounds: wrapBounds
		};
		return output;
	}// prepLoopingPattern()
	
	
	
	/********************	prepSinglePattern()
	Clones a single image,
	 optionally adds a buffer, 
	 figures out wrapping/edge coordinates.
	
	// input
	original_pic
	view_rect
	buffer
	
	// output
	pattern_pic
	wrapBounds
	*******************/
	function prepSinglePattern( original_pic, view_rect, buffer, center_mat )
	{
		var xx,yy,ww,hh;
		var copy_rect, paste_p;
		var view_pic;
		var output, pattern_pic, wrapBounds;
		view_pic = new BitmapData( view_rect.width,view_rect.height, !edgeOpaque,0 );
		//buffer = 2;		// 2 pixel out from each edge
		
		
		
		// prepare the pattern for wrapping
		// ww = original_pic.width +(buffer*2) +view_pic.width;
		// hh = original_pic.height +(buffer*2) +view_pic.height;
		ww = original_pic.width +(buffer*2);
		hh = original_pic.height +(buffer*2);
		pattern_pic = new BitmapData( ww,hh, !edgeOpaque,0 );
		
		// copy the original into the prepped image
		copy_rect = new Rectangle( 0,0, original_pic.width, original_pic.height );
		paste_p = new Point( buffer, buffer );
		pattern_pic.copyPixels( original_pic, copy_rect, paste_p );
		
		
		var halfWidth = Math.floor( original_pic.width / 2 );
		var halfHeight = -center_mat.ty;		// the bottom edge is the rotation point
		wrapBounds = {
			xMin: buffer -halfWidth,
			yMin: buffer -halfHeight,
			xMax: buffer +original_pic.width -halfWidth,
			yMax: buffer +halfHeight
		};
		
		
		output = {
			pattern_pic: pattern_pic,
			wrapBounds: wrapBounds
		};
		return output;
	}// prepSinglePattern()
	
	
	
	// Horizontal movement
	function moveHorz( speed ){
		moveVec_mat.identity();
		moveVec_mat.translate( speed, 0 );
		moveVec_mat.concat( angle_neg_mat );
		position_neg_mat.translate( moveVec_mat.tx, moveVec_mat.ty );
	}// moveHorz()
	
	
	// Vertical movement
	function moveVert( speed ){
		moveVec_mat.identity();
		moveVec_mat.translate( 0, speed );
		moveVec_mat.concat( angle_neg_mat );
		position_neg_mat.translate( moveVec_mat.tx, moveVec_mat.ty );
	}// moveVert()
	
	
	// Rotation movement
	function moveRot( speed ){
		angle_mat.rotate( rot_speed *speed );
		angle_neg_mat.rotate( -rot_speed *speed );
		pos_radian += (rot_speed *speed);
		limitPosRadian();		// limit pos_radian values to be between:  0...2*PI		(radian equivalent of 360 degrees)
	}// moveRot()
	
	
	function wrapAround()
	{
		var xx = -position_neg_mat.tx;
		var yy = -position_neg_mat.ty;
		var hasWrapped = false;
		
		// if:   wrapImage = true
		if( wrapImage ){
			if(xx < wrapCoords.xMin){
				xx = wrapCoords.xMax;
				hasWrapped = true;
			}
			if(xx > wrapCoords.xMax){
				xx = wrapCoords.xMin;
				hasWrapped = true;
			}
			if(yy < wrapCoords.yMin){
				yy = wrapCoords.yMax;
				hasWrapped = true;
			}
			if(yy > wrapCoords.yMax){
				yy = wrapCoords.yMin;
				hasWrapped = true;
			}
		}// if:   wrapImage = true
		
		
		// if:   wrapImage = false
		else{
			if(xx < wrapCoords.xMin){
				xx = wrapCoords.xMin;
				hasWrapped = true;
			}
			if(xx > wrapCoords.xMax){
				xx = wrapCoords.xMax;
				hasWrapped = true;
			}
			if(yy < wrapCoords.yMin){
				yy = wrapCoords.yMin;
				hasWrapped = true;
			}
			if(yy > wrapCoords.yMax){
				yy = wrapCoords.yMax;
				hasWrapped = true;
			}
		}// if:   wrapImage = false
		
		
		if(hasWrapped){
			// reset position to an absolute coordinate
			position_neg_mat.identity();
			position_neg_mat.translate( -xx, -yy );
		}// hasWrapped()
		
	}// wrapAround()
	
	
	// Update the display-image
	function updateImage()
	{
		display_mat.identity();
		display_mat.concat( center_mat );		// start at -5,-5		(to rotate around the center)
		display_mat.concat( position_neg_mat );	// add the position coords
		display_mat.concat( angle_mat );		// add the angle		(to rotate)
		display_mat.concat( center_neg_mat );		// shift +5,+5		(to display the center at the center)
		
		display_mat.concat( crop_mat );		// offset to top-left of crop area
		
		// uses "clipRect" to reduce drawing work
		if( !wrapImage )		crop_pic.fillRect( full_rect, edgeColor );
		// crop_pic.draw( pattern_pic, display_mat );		// the size of crop_pic will automatically crop down to the correct width and height
		crop_pic.draw( pattern_pic, display_mat, null, null, clipRect );		// the size of crop_pic will automatically crop down to the correct width and height
		
		output.broadcastMessage("onUpdate");
	}// updateImage()
	
	
	// limit pos_radian values to be between:  0...2*PI		(radian equivalent of 360 degrees)
	function limitPosRadian(){
		var rad_360 = (Math.PI*2);
		while(pos_radian < 0)		pos_radian += rad_360;
		if(pos_radian > rad_360)		pos_radian %= rad_360;
	}// limitPosRadian()
}// make_loopingImage()